/* main.c - Hello World + PLL + Interrupt example for S32R274 */
/* Description:  Drive CLKOUT0 to external pin with multiple cores */
/* Rev 1.0 Aug 10 2018 D Chung - production version */
/* Copyright NXP Semiconductor, Inc 2016 All rights reserved. */

/*******************************************************************************
* NXP Semiconductor Inc.
* (c) Copyright 2018 NXP Semiconductor, Inc.
* ALL RIGHTS RESERVED.
********************************************************************************
Services performed by NXP in this matter are performed AS IS and without
any warranty. CUSTOMER retains the final decision relative to the total design
and functionality of the end product. NXP neither guarantees nor will be
held liable by CUSTOMER for the success of this project.
NXP DISCLAIMS ALL WARRANTIES, EXPRESSED, IMPLIED OR STATUTORY INCLUDING,
BUT NOT LIMITED TO, IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR
A PARTICULAR PURPOSE ON ANY HARDWARE, SOFTWARE ORE ADVISE SUPPLIED TO THE PROJECT
BY NXP, AND OR NAY PRODUCT RESULTING FROM NXP SERVICES. IN NO EVENT
SHALL NXP BE LIABLE FOR INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF
THIS AGREEMENT.

CUSTOMER agrees to hold NXP harmless against any and all claims demands or
actions by anyone on account of any damage, or injury, whether commercial,
contractual, or tortuous, rising directly or indirectly as a result of the advise
or assistance supplied CUSTOMER in connection with product, services or goods
supplied under this Agreement.

Copyright 2018 NXP.  This software is owned or controlled by NXP and may only
be used strictly in accordance with the applicable license terms.  By expressly
accepting such terms or by downloading, installing, activating and/or otherwise
using the software, you are agreeing that you have read, and that you agree to
comply with and are bound by, such license terms.  If you do not agree to be
bound by the applicable license terms, then you may not retain, install, activate
or otherwise use the software.
********************************************************************************
* File              main.c
* Owner             David Chung
* Version           1.0
* Date              Aug-2-2018
* Classification    General Business Information
* Brief             S32R274 cores produces different CLKOUTs and takes turns
* 					controlling the pin
********************************************************************************
* Detailed Description:
* S32R274's default clock source is 16 MHz IRCOSC.  This code example
* configures the PLL0 to 240 MHz, PLL1 to 160 MHz, and SDPLL to zzz MHz.
* The system clock selector selects PLL0_PHI_CLK as its source. The clock
* domain SYS_CLK divides that by 3, such that SYS_CLK = 60 MHz.
* Each of S32R's three cores configures a different clock source
* for the CLKOUT0 pin. The z4 core drives PLL0_PHI_CLK out divided by 20
* (240 MHz/20 = 12 MHz), the z7a core drives PLL1_PHI_CLK divided by 20
* (160 MHz/20 = 8 MHz), and z7b drives XOSC divided by 20 (20 MHz/20 = 1MHz).
* The cores take turns controlling the CLKOUT0 pin my means of a
* PIT interrupt. The PIT is configured to interrupt every 5 seconds.
* The PIT interrupt fires in a round robin fashion amongst the cores
* such that each core's PIT interrupt routine is executed one after
* another, wherein the CLKOUT0 clock source changes. The switch is
* indicated in the oscilloscope as well as the user LED.
*
* Each core will configure a different GPIO to toggle.  When these GPIO
* pins are connected to user LEDs, you can observe a toggle.  The
* frequency of the toggle is based on an internal counter.
*
* Refer to "Z4 Waveform.png", "Z7a Waveform.png", and "Z7b Waveform.png"
* for the expected output of each core.
*
* NOTE: Make sure to start running the Z7a and Z7b threads before
* starting the Z4 thread.
*
* ------------------------------------------------------------------------------
* Test HW:         S32R274RRUEVB + MPC57XXMB
* MCU:             S32R274
* Terminal:        None
* Fsys:            240 MHz PLL
* Debugger:        USB Multilink
* Target:          FLASH
* EVB connection:  Connect pin P9.7 (CLK_OUT0) to an oscilloscope to check
* 					CLK_OUT0 frequency. Connect P8.1 (PA0) to LED P7.1 (D2),
* 					P8.2 (PA1) to P7.2 (D3), and P8.3 (PA2) to P7.3 (D4).
*
* 					Refer to "Hello_World_PLL_Interrupt_Hardware_Setup.jpg"
*
********************************************************************************
Revision History:
Version  Date         Author  			Description of Changes
1.0      Aug-10-2018  David Chung	  	Initial version

*******************************************************************************/

#include "derivative.h" /* include peripheral declarations */

#define KEY_VALUE1 0x5AF0ul
#define KEY_VALUE2 0xA50Ful

#define COUNTER_MAX 500000

extern void xcptn_xmpl(void);

void CLK_Init_Z4(void);

void CLK_Init_Z7a(void);

void CLK_Init_Z7b(void);

void CLKOUT0_Init_Z4(void);

void Peripherals_Init(void);

void PIT_Init(void);

void PIT_InterruptHandler(void);

void hw_init(void)
{
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	uint32_t mctl = MC_ME.MCTL.R;
#endif
#if defined(TURN_ON_CPU1)
	/* enable core 1 in all modes */
	MC_ME.CCTL2.R = 0x00FE;
	/* Set Start address for core 1: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR2.R = 0x1080000 | 0x1;
#else
    MC_ME.CADDR2.R = 0x4006a800 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif	
#if defined(TURN_ON_CPU2)
	/* enable core 2 in all modes */
	MC_ME.CCTL3.R = 0x00FE;
	/* Set Start address for core 2: Will reset and start */
#if defined(START_FROM_FLASH)
    MC_ME.CADDR3.R = 0x1100000 | 0x1;
#else
    MC_ME.CADDR3.R = 0x400d5000 | 0x1;
#endif /* defined(START_FROM_FLASH) */
	
#endif
#if defined(TURN_ON_CPU1) || defined(TURN_ON_CPU2)
	MC_ME.MCTL.R = (mctl & 0xffff0000ul) | KEY_VALUE1;
	MC_ME.MCTL.R =  mctl; /* key value 2 always from MCTL */
#endif
}

void CLK_Init_Z4(){
	/* Configure PLL0 to 240 MHz */
	CGM.AC3_SC.R = 0x01000000; //Select XOSC (40 MHz) as source of PLL0

	/* f_PLL0_PHI0 = fin * (MFD/(PREDIV*RFDPHI))
	 * 				= 40 * (12/(2*1))
	 * 				= 240
	 *
	 * 	f_PLL0_PHI1 = fin * (MFD/(PREDIV*RFDPHI1))
	 * 				= 40 * (12/(2*6))
	 * 				= 40
	 */
	PLLDIG.PLL0DV.R = 0x3001200C;

	/* Turn on clock sources */
	MC_ME.DRUN_MC.R = 0x00130072; //Turn on IRC, XOSC, and PLL0. Set system clock to PLL0_PHI_CLK (240 MHz)

	/* Perform mode switch for changes to take effect*/
	MC_ME.MCTL.R = 0x30005AF0;
	MC_ME.MCTL.R = 0x3000A50F;
	while(MC_ME.GS.B.S_MTRANS); //Wait for transition to complete

	/* Configure SYS_CLK */
	CGM.SC_DC0.R = 0x80030000; //Enable and divide by 4 (60 MHz)

	/* Configure PBRIDGEx_CLK */
	CGM.SC_DC3.R = 0x80030000; //Enable and divide by 4 (60 MHz)
}

void CLK_Init_Z7a(){
	/* Select PLL0_PHI1_CLK as source of PLL1 */
	CGM.AC4_SC.B.SELCTL = 0;

	/* f_PLL1_PHI0 = fin * (MFD/(2*RFDPHI))
	 * 				= 40 * (16/(2*2))
	 * 				= 160
	 */
	PLLDIG.PLL1DV.R = 0x00020010;

	/* Turn on clock sources */
	MC_ME.DRUN_MC.R = 0x001300F2; //Turn on IRC, XOSC, PLL0, and PLL1. Set system clock to PLL0_PHI_CLK (240 MHz)

	/* Perform mode switch for changes to take effect*/
	MC_ME.MCTL.R = 0x30005AF0;
	MC_ME.MCTL.R = 0x3000A50F;
	while(MC_ME.GS.B.S_MTRANS); //Wait for transition to complete
}

void CLK_Init_Z7b(){
	/* Placeholder for now.
	 * TODO: Configure SDPLL
	 */
}

void CLKOUT0_Init_Z4(){
	/* S32R27 boots from the 16 MHz IRCOSC.
	 * MCB controls CLKOUT source selects.
	 */

	MCB.CLKOUT_SEL.B.CLKOUT0 = 0b0010; //Select PLL0 for CLKOUT0
	CGM.AC14_DC0.R = 0x80130000; //Divide by (19+1) = 20 for output of 12 MHz

	/* CLKOUT0 maps to PB6 (MSCR22) */
	SIUL2.MSCR[22].B.SSS = 1; //Configure pin for CLKOUT0
	SIUL2.MSCR[22].B.OBE = 1; //Enable output buffer

	/* Configure GPIO PA0 (P8.1) */
	SIUL2.MSCR[0].B.OBE = 1;
	SIUL2.GPDO[0].B.PDO_4n = 1;

	/* Disable all other GPIOs */
	SIUL2.GPDO[0].B.PDO_4n1 = 1;
	SIUL2.MSCR[1].B.OBE = 0;

	SIUL2.GPDO[0].B.PDO_4n2 = 1;
	SIUL2.MSCR[2].B.OBE = 0;
}

void Peripherals_Init(){
	MC_ME.RUN_PC[0].R = 0x00000000; //Disable peripherals following RUN_PC0 in all power modes
	MC_ME.RUN_PC[1].R = 0x000000FE; //Enable peripherals following RUN_PC1 in all power modes

	MC_ME.PCTL[30].R = 0x01; //Select RUN_PC1 for PIT_RTC0

	/* Perform mode change for config to take effect */
	MC_ME.MCTL.R = 0x30005AF0;
	MC_ME.MCTL.R = 0x3000A50F;
	while(MC_ME.GS.B.S_MTRANS); //Wait for transition to complete
}

void PIT_Init(){
	PIT_0.MCR.B.MDIS = 1; //Disable the module

	/* Configure PIT_0 Ch0 */
	PIT_0.TIMER[0].LDVAL.R = 300000000;	//PIT clock source is PBRIDGEx_CLK (60 MHz). Have channel expire every 5 second
	PIT_0.TIMER[0].TCTRL.R = 0x00000003; //Enable PIT_0 Ch0 interrupt and enable the channel

	/* Enable PIT in the INTC */
	INTC.PSR[226].R = 0x400F;  //Configure PIT_0 Ch0 interrupt in INTC. Set priority 15 and send to Core 1 (Z7a)

	/* Enable the PIT */
	PIT_0.MCR.B.MDIS = 0; //Reenable the module
}

void PIT_InterruptHandler(){
	/* Clear the interrupt flag */
	PIT_0.TIMER[0].TFLG.R = 0x00000001; //w1c

	/* Configure CLKOUT to use PLL0_PHI0_CLK */
	CLKOUT0_Init_Z4();

	/* Configure PIT interrupt so next time it fires,
	 * it goes to Processor 1, the Z7a core.
	 */
	INTC.PSR[226].R = 0x400F;

}

int main(void)
{
	int counter = 0;

	xcptn_xmpl ();              /* Configure and Enable Interrupts */

	/* Configure clock for all 3 cores */
	CLK_Init_Z4();				/* Configure PLL0 to 240 MHz and SYS_CLK = PLL0_PHI0 */

	CLK_Init_Z7a();				/* Configure the clock for z7a. PLL1 to 160 MHz */

	CLK_Init_Z7b();				/* Configure the clock for z7b. SDPLL to xxx MHz */

	/* Configure CLKOUT and peripherals */
	CLKOUT0_Init_Z4();			/* Configure CLKOUT0 pin */

	Peripherals_Init();			/* Enable all relevant peripherals */

	PIT_Init();					/* Configure the PIT */

	/* Loop forever */
	for(;;) {	   
	   	counter++;

	   	/* Wait 500000 counts of internal counter
	   	 * before toggling GPIO.
	   	 */
	   	if(counter >= COUNTER_MAX){
	   		SIUL2.GPDO[0].B.PDO_4n ^= 1; //Toggle PA0
	   		counter = 0; //Reset counter
	   	}
	}
}


